#Title: halfdawn_function.R 
#Type: Script
#Version: 1.0
#Date: 2014-03-07
#Author: Martial Sankar
#Maintainer: Martial Sankar <martial.sankar@unil.ch>
#Description: Suite of function for automated image pre-processing and analysis of hypocotyl cross-sections.
#License: GPL (>= 2)
#Depends: R (>= 2.14.0)
#OS_type: unix
#Depends on: EBImage, RColorBrewer
#URL: http://www3.unil.ch/wpmu/hardtkelab/


# getBinImages function, returns binarized & sliced images in tif format, take as input :
#              @ saveRaw, Boolean, save quarter images
#              @ raw, String, raw input images
#              @ saveThresh, Boolean, save thresholded images
#              @ nslice, Integer, number of slice 4(default)
getBinImages <- function(raw, saveRaw=TRUE, saveThresh=TRUE, nslice=4, das=NULL, gamma=15, contrast = 8)
{
	xyl0 <- readImage(raw)
	ut <- untile(xyl0, c(nslice/2, nslice/2))
	slList <- list()
	for (i in 1 : nslice){
		slList[[i]] <- ut[,,i]
	}
	
	tp <- lapply(1:nslice, getBinByImage, slicedIm=slList,saveThresh=saveThresh, fn=raw, das=das, gamma=as.numeric(gamma), contrast = as.numeric(contrast))
	if(saveThresh){cat("# Saved Thresholded Images ::\n")}
	
	tp <- combine(tp)
	im <- tile(tp, nx=2, lwd=0)
	
	if(saveThresh){
		savedn <- paste(gsub("\\.tif","", raw),"_binarized.tif",sep="")
		writeImage(im, savedn)
		cat("## ", savedn, "\n")
	}
	
	return(im)
	
}


#getBinByImage function returns the thresholded image (or slice), take as input :
#             @slicedIm, list, containing Image Objects
#             @saveThresh, Boolean, save Threshed picture
#             @fn, String, raw filename
#             @das, Int, Age of the section
getBinByImage <- function(i,slicedIm, saveThresh,fn, das=NULL, gamma=NULL, contrast = NULL)
{
	if(is.character(slicedIm)){
		imCur <- readImage(slicedIm)
		fn <- slicedIm
		i <- 0
	}else {imCur <- slicedIm[[i]]}
	
	if (is.null(das)){
		gammaCorr <- gamma
		contrast <-  contrast
	}else {
		das <- as.numeric(das)
		if(das<=35){
			gammaCorr <- 3
			contrast <- 13
		}else {
			gammaCorr <- 15
			contrast <- 8
		}
	}
	
	if(i == 1){
		cat("#Operation :: Gamma Correction:",gammaCorr ," , Contrast:", contrast,"\n" )
	}
	
	xyl0Inv <- 1-imCur # invert LUT
	xyl0Invc <- xyl0Inv^gammaCorr #  gamma correction
	xyl0Invb <- contrast*xyl0Invc #  contrast
	xylthresh <- thresh(xyl0Invb, w=30, h=30, offset=0.0)
	
	# re-invert LUT
	xylthresh[xylthresh==1] <- -1
	xylthresh[xylthresh==0] <- 1
	xylthresh[xylthresh==-1] <- 0
	xyllabFilled <- fillHull(xylthresh) # fill gap
	
	# Morphological op
	xylerode <- erode(xyllabFilled, makeBrush(5, shape = "gaussian", sigma=10))
	xylerode <- erode(xylerode, makeBrush(4, shape = "gaussian", sigma=10)) # 3
	xyldilate <- dilate(xylerode,makeBrush(3, shape = "gaussian", sigma=10))
	xylfilled2 <- fillHull(xyldilate)
	
	return(xylfilled2)
}


#getFeaturesPerRay function that returns list of cell features per Ray, take as input
#            @ imSeg, String or Image Object, Segmented images (ImageJ output)
#            @ imRaw, String or Image Object, Raw image (used to generate overlayed segmentation images)
#            @ saveOverlay, String, path & name to save the overlayed images
#            @ saveGraph, String, prefix path & name to generate graph per rays
#            @ rayThresh, Numeric, filters all objects that have higher r
#            @ checkRay, String, prefix path and name to save images with highlighted ray
getFeaturesPerRay <- function(imSeg, imRaw=NULL, saveOverlay, saveGraph, rayThresh, angle, angleTol=0.05,checkRay,rotation=0)
{
	if(is.character(imSeg)){
		imCur <- readImage(imSeg)
		fn <- imSeg
		cat("# Processing *",fn,"*\n")
	}else {imCur <- imSeg}
	
	labs<- bwlabel(imCur)
	labs <- labs[,,1]
	colorMode(labs) <- Grayscale
	
	cat("## Number of Objects : ", max(labs), "\n")
	
	if(is.null(imRaw)){
		saveOverlay <- FALSE
	}else if (is.character(saveOverlay)){
		imRawCur <- readImage(imRaw)
		fnRaw <- imRaw
		imRawCur <- channel(imRawCur, "rgb")
		imOv <- paintObjects(x=labs,tgt=imRawCur, col="#00FF00")
		outov <- paste(saveOverlay,"_overlay.tif",sep ="")
		writeImage(imOv, outov, quality = 100)
		cat("## Overlay : ", outov, "\n")
	}
	
	#  compute features global & convert to polar coordinates
	
	if(rotation>0){
		labs <- rot90(labs,rotation)
	}
	ftrs <- computeFeatures.shape(labs)
	ftrsMom <- computeFeatures.moment(labs)
	
	radiusV <- sqrt(ftrsMom[,1]^2+ftrsMom[,2]^2)
	angleV <- asin(ftrsMom[,2]/radiusV)
	polarCoord <- as.matrix(cbind(radiusV, angleV))
	# rayThresh
	ix <- as.numeric(which(polarCoord[,1]<rayThresh), arr.ind =T)
	polarCoord <- polarCoord[ix,]
	ftrs <- ftrs[ix,]
	rm(ix)
	# extract features per ray
	obj <- getRayFeature(angle =  angle, ctrCrdMat = polarCoord, shpCrdMat = ftrs,
			angleTol = angleTol,
			rayThresh = rayThresh,
			imSeg= labs ,
			checkRay=checkRay ,
			saveGraph= saveGraph)
	
	return(obj)
	
}

#_3 ExtractFeatures per Ray

# getFeatPerRay function returns rays description
#              @ angle, Numeric, along the cells are extracted (in radians)
#              @ ctrCrdMat, Matrix, contains center position (*polar* coordinates)
#              @ shpCrdMat, Matrix, contains shape information
#              @ rayThresh, Numeric, in pixels
#              @ angleTol, Numerix, angle tolerance (in radians)
#              @ imSeg, Image Object, the segmented images (used to check the rays)
getRayFeature <- function(angle, ctrCrdMat, shpCrdMat,  rayThresh, angleTol=0.05, imSeg, checkRay,saveGraph)
{
	#  1 get object IDs
	idx <- as.numeric(which(ctrCrdMat[,2]<(angle+angleTol) & ctrCrdMat[,2]>(angle-angleTol)))
	#  2 create index
	ix4viz <- NULL
	ftrstot <- NULL
	ftrsMomtot <- NULL
	polarCoordTot <- NULL
	for (id in idx)
	{
		df <- which(imSeg== as.numeric(row.names(shpCrdMat)[id]), arr.ind=TRUE)
		ftrsi <- shpCrdMat[as.numeric(id),]
		#ftrsMomi <- shpCrdMat[as.numeric(id),]
		polarCoordi <- ctrCrdMat[as.numeric(id),]
		
		ix4viz <- rbind(ix4viz,df)
		ftrstot <- rbind(ftrstot,ftrsi)
		#ftrsMomtot <- rbind(ftrsMomtot,ftrsMomi)
		polarCoordTot <- rbind(polarCoordTot,polarCoordi)
	}
	
	# check by visualisation
	
	if(is.character(checkRay)){
		cols <- c("white", rep("black",max(imSeg)))
		cols <-matrix(cols[1+imSeg],nrow=dim(imSeg))
		colsSelect <- cols
		colsSelect[ix4viz] <- rgb(255,100,100, 55, maxColorValue=255)
		iNew <- Image(colsSelect)
		checkRaynm <- paste(checkRay,"_", ceiling(angle),".tif",sep="")
		writeImage(iNew, checkRaynm, quality = 100)
		cat("## check ray in : ", checkRaynm, "\n")
	}
	
	#  plot Graph
	if(is.character(saveGraph)){
		saveGraphnm <- paste(saveGraph,"_bubbleChart_",ceiling(angle),".pdf",sep="")
		pdf(saveGraphnm, height=10, width=13)
		plot(polarCoordTot[,1],  (polarCoordTot[,2]-angle),type = "h", pch=".", cex=2,  xlab="radius", ylab="angle")
		points(polarCoordTot[,1],(polarCoordTot[,2]-angle), cex = ftrstot[,1]/1000, col = rgb(255,100,100, 55, maxColorValue=255), pch=16)
		cat("## graph saved in : ", saveGraphnm, "\n")
		dev.off()
	}
	
	#_ 3 return matrix
	tabFinal <- cbind(polarCoordTot, ftrstot)	
	return (tabFinal)
	
}



# vizClassRes, function that returns segmented images with a mask of the labelled cells
#            @ imSeg,
#            @ labM, matrix, contains ids, features and label for each cells
#            @ rotation
#            @ savePref, Boolean, save resulting  images
vizClassRes <- function(imSeg, labM, rotation, suffix=NULL,savePref=TRUE)
{
	if(is.character(imSeg)){
		imCur <- readImage(imSeg)
		fn <- imSeg
		cat("# Processing *",fn,"*\n")
	}else {imCur <- imSeg}
	
	labs<- bwlabel(imCur)
	labs <- labs[,,1]
	colorMode(labs) <- Grayscale
	
	cat("## Number of Objects : ", max(labs), "\n")
	
	# identify rotation of interest using the ids
	
	idsall <- row.names(labM)
	vrot <- unlist(lapply(strsplit(idsall, "\\."), function(v){v[1]}))
	vlabCell <- unlist(lapply(strsplit(idsall, "\\."), function(v){v[2]}))
	
	ixcurr <- which(vrot==as.character(rotation))
	vlabCellcurr <- vlabCell[ixcurr]
	labMcurr <- labM[ixcurr,]
	
	if(rotation>0){
		labs <- rot90(labs,rotation)
	}
	
	#  extract index for each label
	
	labels <- sort(unique(labMcurr[,ncol(labMcurr)]))
	lix <- list()
	
	for (i in 1:length(labels)){
		idsCurr <- vlabCellcurr[which(labMcurr[,ncol(labMcurr)]==labels[i])]
		ix4viz <- NULL
		for (id in idsCurr){
			df <- which(labs==as.numeric(id), arr.ind=TRUE)
			ix4viz <- rbind(ix4viz,df)
		}
		
		lix[[i]] <- as.matrix(ix4viz)
	} # OR USE split
	
	cat("# Generating mask Images ...")
	colPal <- brewer.pal(8, "Accent")
	cols <- c("black", rep("white",max(labs)))
	cols <-matrix(cols[1+labs],nrow=dim(labs))
	colsSelect <- cols
	
	for (i in 1:length(lix)){
		colsSelect[lix[[i]]] <- colPal[i]
	}
	
	cat("# Done\n")
	
	iNew <- Image(colsSelect)
	
	cat("# Creating Image... \n")
	
	if(savePref==TRUE){
		
		if(is.null(suffix)){
			suffixN <- "checkGroup.tif"
		}else {
			suffixN <- paste( suffix,"_checkGroup.tif", sep ="")
		}
		checkgrpnm <-  paste(gsub("colBasins.tif","", imSeg) ,suffixN,sep="")
		writeImage(iNew, checkgrpnm, quality = 100)
		cat("## saved in : ", checkgrpnm, "\n")
	}
	
}


#saveOverlayImage, function that returns and save the overlay between the original images and the segmented one.
#            @imSeg, String or Image Object, Segmented images (ImageJ output)
#            @imRaw, String or Image Object, Raw image (used to generate overlayed segmentation images)
saveOverlayImage <- function(imSeg, imRaw){
	
	imCurr <- opIm(imSeg)
	labs<- bwlabel(imCurr)
	labs <- labs[,,1]
	colorMode(labs) <- Grayscale
	rm(imCurr)
	if(is.null(imRaw)){
		cat(" Raw image required!\n")
		return(NULL);
	}else {	
		imRawCur <- opIm(imRaw)
		if(is.character(imRaw)){
			fnRaw <- imRaw
			saveOverlay<- gsub(".tif", "_saveOverlay.tif",imRaw)
		}else {saveOverlay <- gsub(".tif", "_saveOverlay.tif",imSeg)}
		#imRawCur <- imRawCur[,,1]
		
		imRawCur <- channel(imRawCur, "rgb")
		imOv <- paintObjects(x=labs,tgt=imRawCur, col="#00FF00")
		writeImage(imOv, saveOverlay, quality = 100)
		cat("## Overlay : ", saveOverlay, "\n")
	}
	
}



#opIm, function that open Images, input can be an image object or a filename
#      @imSeg, String or ImageObject, path name of the image file or imageObject
opIm <- function(im){
	
	if(is.character(im)){
		imCur <- readImage(im)
	}else {imCur <- im}
	
	return(imCur);
	
}


