%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Matlab code of the PT-SGR approach developed in:

% Laloy E., N. Linde, D. Jacques, and G.A. Mariethoz (2016), Merging parallel 
% tempering with sequential geostatistical resampling for improved posterior 
% exploration of high-dimensional subsurface categorical fields, 
% Advances in Water Resources, 90, 57-69, doi:10.1016/j.advwatres.2016.02.008.

% Author: Eric Laloy <elaloy@sckcen.be>, February 2016

% Remark: 
% In the paper we used the 2015 C executable version of the DeeSse (DS) MPS
% algorithm by Mariethoz et al. (2010) to generate proposals in the Markov 
% chain Monte Carlo (MCMC) sampling. Since the DS input files have slightly 
% changed since 2015, this PT-SGR implementation is for the 2017 DS version.
% The DS code is not provided herein. To get a DS license, please contact 
% Julien Straubhaar <julien.straubhaar@unine.ch>.
% For this PT-SGR version to run, you need to have in the same directory as 
% the Matlab code: the deesse.exe and the two dll files together with a
% folder named 'lic' that contains your license file.
                                                                                            
% Copyright (C) 2016  the authors                                                      
                                                                                            
% This program is free software: you can modify it under the terms of the 
% GNU General Public License as published by the Free Software Foundation, 
% either version 3 of the License, or (at your option) any later version.                                           
                                                                                      
% This program is distributed in the hope that it will be useful, but WITHOUT 
% ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
% FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
% more details.                                      

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clc;clear;
% Initialization

type='mt19937ar'; % Set random number generator
rng(sum(100*clock),type);

% Data
example=1;
addpath([pwd '\example_' num2str(example)]);
Model_dir = [pwd '\example_' num2str(example) '\MaFloT'];
addpath(Model_dir);
if example==1
    load head_measurements head_data_c stderr;
    data.obs=head_data_c(:);
end
data.std=stderr;

mcmc.example=example; clear example;

% General mcmc settings
mcmc.nc=16; % Number of parallel cores = number of chains
mcmc.nt=mcmc.nc; % Will be overridden if parallel tempering is activated
mcmc.temp=ones(1,mcmc.nt); % Will be overridden if parallel tempering is activated
mcmc.it_max=5e4; % Maximum number of (parallel) iterations in each chain
mcmc.iter=1; % Iteration number in each chain
mcmc.fe=mcmc.nt; % Number of forward evaluations
mcmc.it_write = 50 ; % Number of iteration in each chain for writing to disk: not less than mcmc.it_update for the case a restart is needed
mcmc.target_ar = 23; % Target acceptance rate (%)
mcmc.it_update = 50 ; % Number of iterations for computing the ar and updating the conditioning fraction mcmc.phi (see further)
mcmc.it_update_max = 2500 ; % Maximum nulber of iterations

% Parallel tempering settings
mcmc.DoPT='Yes'; 
% If set to 'Yes', then parallel tempering is activated.
% Otherwise, standard SGR sampling is performed.
if strcmp(mcmc.DoPT,'Yes');
    tlev=16;
    tmax=6;
    t=exp(linspace(log(1),log(tmax),tlev));
    %t=[1 1 t];
    nt=length(t);
    mcmc.temp=t';
    mcmc.nt=nt; clear nt;
    mcmc.pt_swap='Random'; % Random, Neighbor, or Neighbor_all
    mcmc.proba_swap=1;
    mcmc.seq_temp1=[1:1:length(mcmc.temp)];
    mcmc.seq_temp1=mcmc.seq_temp1(mcmc.seq_temp1==1);
    output.ar_swap=zeros(floor(mcmc.it_max/mcmc.it_update),1+mcmc.nt);
    mcmc.i_swap_step=0; 
end
output.ar_isr=zeros(floor(mcmc.it_max/mcmc.it_update),1+mcmc.nt);
output.hist_phi=zeros(floor(mcmc.it_max/mcmc.it_update),1+mcmc.nt);
output.store_dir=pwd;

Init='Yes'; % 'Yes' means that the mcmc.nc input files do not exist yet

mcmc.nx=100; % Dimensions of the grid
mcmc.ny=100;
mcmc.nz=1;

addpath([pwd '\trirnd']);
% mcmc.cond_type:
% 1: Fraction of conditioning data = non-resimulated area) 
% or 2: side length of box-type resimulated area
mcmc.cond_type=2; 
if mcmc.cond_type==1 % Initial value of phi
    mcmc.use_phi_dist='Fixed'; % Uniform, Triangular, Gaussian, None,Fixed
    phi_fixed=ones(1,mcmc.nt).*0.01;
    for i=1:mcmc.nt
            mcmc.phi_cv='None';
            mcmc.phi(i)=phi_fixed(i);
            mcmc.phi_max(i)=0.75;
            mcmc.phi_min(i)=0.01;
    end
elseif mcmc.cond_type==2
    mcmc.use_phi_dist='None';
    phi_min=ones(1,mcmc.nt).*11;
    phi_fixed=ones(1,mcmc.nt).*50;
    for i=1:mcmc.nt
        mcmc.phi(i)=phi_fixed(i);
        mcmc.phi_max(i)=50;  
        mcmc.phi_min(i)=phi_min(i); 
    end
elseif mcmc.cond_type==3
    mcmc.use_phi_dist='None'; 
    phi_min=ones(1,mcmc.nt).*22;
    for i=1:mcmc.nt
        mcmc.phi(i)=70;
        mcmc.phi_max(i)=80; 
        mcmc.phi_min(i)=phi_min(i); 
    end
end
load('K_true.mat');mcmc.m_ref=K;clear K; % Reference model for example 1, not used for the inversion

mcmc.Restart='No'; % Is this a restart? Yes or No
data.lik_type=2; % Gaussian log-likelihood function

% Setup parallelization, depends on the Matlab version
DoParallel='Yes';
if verLessThan('matlab','8.2'),
    version_MATLAB = 'old';
else
    version_MATLAB = 'new';
end
if strcmp(DoParallel,'Yes');
    if strcmp(version_MATLAB,'old')
        if(matlabpool('size'))<=0
            matlabpool('open','local',mcmc.nc);
        end
    else
        clear poolobj used_pool
        poolobj = gcp('nocreate');
        if isempty(poolobj)
            used_pool=parpool('local',mcmc.nc);
        end
    end
end
[mcmc, output ] = ptsgr( mcmc, output, data, Init );

eval(['save out.mat mcmc output data']);

if strcmp(version_MATLAB,'old')
    if mcmc.nc > 1
        matlabpool close
    end
end
