% Matlab script to analyzed combined 19F and 1H images
% the script does not include (vendor-dependent) data loading into Matlab
% this script is organized into cells
% interpolate, crop, overlay, segment, quantify
% (c) Ruud van Heeswijk
% if used, please cite: Croci et al. Sci Trans Med 2022


clear all
close all
clc

%the variables below will be used for data saving
scandate='20250713';
mousename='41817ABL';

FoV1H=100; %check both (in mm)
FoV19F=60;


MetModelFlag=0; %flag for metastatis model - different #1H slices

%% --------------------------------------------------------------------------
% load 1H image stack as image1H, crop
% initial data loading will depend on your scanner manufacturer
%-------------------------------------------------------------------------- 
image1H=squeeze(dicomread(H1DicomFolderName)); %example

% crop outer slices that are not covered by 19F
[dimx1H,dimy1H,dimz1H]=size(image1H);
image1H = double(fCropNdArray(image1H,[dimx1H,dimy1H,16]));

%% --------------------------------------------------------------------------
% load 19F raw data as im_FT, crop, interpolate, crop
%--------------------------------------------------------------------------    

im_FT_crop=squeeze(dicomread(YourF19DicomFolderName)); %example  image stack loading command
[dimx19F,dimy19F,dimz19F]=size(im_FT_crop);

% flip 19F images with 1H FOV if needed
im_FT_crop = flip(im_FT_crop,1); % if needed
im_FT_crop = flip(im_FT_crop,2); % if needed
im_FT_crop = flip(im_FT_crop,3); % if needed


% interpolate and crop 19F images to 1H size, remove slice oversampling
H1InterpFactor=dimx19F/(dimx1H*FoV1H/FoV19F);

[Xq,Yq,Zq] = meshgrid(0.1:H1InterpFactor:dimx19F,0.1:H1InterpFactor:dimx19F,0.1:dimz19F);
[Xo,Yo,Zo] = meshgrid(0.1:128,0.1:128,0.1:16);

image19FFT = interp3(Xo,Yo,Zo,abs(im_FT_crop),Xq,Yq,Zq);

image19FFT = fCropNdArray(image19FFT,[dimx1H,dimy1H,dimz1H])*1e6; % also scale image


clear Xo Yo Zo Xq Yq Zq im_CS im_FT %clear memory

climsFT=[0 max(image19FFT(:))/6]; % set display range


% create MIP images for inspection
mip19FFTCor=max(image19FFT,[],3);
mip19FFTSag=rot90(squeeze(max(image19FFT,[],2)));

mip1HCor=max(image1H,[],3);
mip1HSag=rot90(squeeze(max(image1H,[],2)));


if exist('hotiron')==0
    load('MyColormaps','hotiron'); %load custom colormap
end

%get noise standard deviation
NoiseRoiFT=abs(im_FT_crop(1:dimz1H,1:100,:))*1e6; %use data before interpolation, but do scale
NoiseStDevFT=std(NoiseRoiFT(:));

%% -------threshold 19F ---------------
%create tresholded transparancy masks for the overlay images
[dim1,dim2,dim3]=size(image19FFT);
imAlphaDataFT=zeros(dim1,dim2,dim3);
imAlphaDataFT(image19FFT>NoiseStDevFT*6)=1; 

mipimAlphaDataFT=max(imAlphaDataFT,[],3);

% plot 19F mips, masks and overlays to check noise and signal location
figure(32), title ('CHECK SIGNAL LOCATION'), hold off
subplot(1,2,1),imagesc(mip19FFTCor,climsFT), axis image off %climsFT
subplot(1,2,2),imagesc(mipimAlphaDataFT), axis image off

figure(33), title ('CHECK SIGNAL LOCATION'), hold off
bgrgb=repmat(mip1HCor/max(mip1HCor(:)),[1 1 3]);
subplot(1,2,1),imbg1=imshow(bgrgb,[0 0.5],'InitialMagnification','fit');
hold on
overlayscaling=0.15;
imoverlay1=imagesc(mip19FFTCor,[0 max(image19FFT(:))]*overlayscaling); 
set(imoverlay1,'AlphaData',ones(size(mip19FFTCor))*0.3);
axis equal tight off

bgrgb=repmat(mip1HSag/max(mip1HSag(:)),[1 1 3]);
subplot(1,2,2),imbg2=imshow(bgrgb,[0 0.5],'InitialMagnification','fit');
hold on
imoverlay2=imagesc(mip19FFTSag,[0 max(image19FFT(:))]*overlayscaling); %
set(imoverlay2,'AlphaData',ones(size(mip19FFTSag))*0.3); 
axis off, daspect([dimx1H/60 1 1])
% colormap jet

%% overlay images
if ishandle(1)
    close(1)
end
if ishandle(2)
    close(2)
end

climsFT=[0 max(image19FFT(:))]*0.2;

%in case of misalignment of 19F and 1H (check reference tube and liver)
upshift=0; % shift overlaid 19F image up
leftshift=0; %shift overlaid 19F image left
skipslice19F=0; %differentiate the overlaid 19F image
skipslice1H=0; %differentiate the overlaid 19F image

%look at overlay
for ctr=1:dim3-skipslice19F-skipslice1H
    figure(21) 
    bgrgb=squeeze(image1H(:,:,ctr+skipslice1H));
    bgrgb=repmat(bgrgb/max(image1H(:)*0.6),[1 1 3]);
    imbg2=imshow(bgrgb,[0 0.5],'InitialMagnification','fit');
    hold on
    colormap(hotiron)%'hot')
    colorbar, colorbar('southoutside')
    imoverlay=imagesc(squeeze(image19FFT(:,:,ctr+skipslice19F)/NoiseStDevFT),'XData',[1-leftshift (dim1-leftshift)/1.001],'YData',[1-upshift (dim2-upshift)/1.0001],[0 15*NoiseStDevFT]);
    set(imoverlay,'AlphaData',0.4*squeeze(imAlphaDataFT(:,:,ctr+skipslice19F)));
    axis equal tight off
    pause%(0.5)
end


%% look at 1H image to identify slices that have tumor in them (with your viewer of preference)
 

%% fill out tumor slice range, then segment tumor
startslice=8;
stopslice=14;
if MetModelFlag==0
    clims1H=[250 700]; % 1H display range
elseif MetModelFlag==1
    clims1H=[250 2250]; % 1H display range
end

if ishandle(1)
    close(1)
end

maskTumor1H=zeros(dim1,dim2,dim3);
for ctr=startslice:stopslice
    figure(54),imagesc(squeeze(image1H(:,:,ctr)),clims1H), axis image, colormap gray, title('SEGMENT THE TUMOR')
    tempmask=roipoly;
    maskTumor1H(:,:,ctr)=tempmask;
pause
end

%% define rough (VERY inclusive) ROIs that go through the 19F image stack for the reftube/liver/tumor

close(34,343)
mip19FFTCor=max(image19FFT,[],3);
figure(34)
imagesc(mip19FFTCor,[0 max(image19FFT(:))]), axis image off
title('SEGMENT THE REFERENCE TUBE')

% create the reference tube mask
maskReferenceFT=roipoly;
maskReferenceFT=repmat(maskReferenceFT,[1 1 dim3]);

% put the upper 90% of the values (to avoid ghosts) in an array
arrayReferenceFT=image19FFT(maskReferenceFT==1);
arrayReferenceFT=arrayReferenceFT(arrayReferenceFT>0.1*max(arrayReferenceFT));
figure(35),hist(arrayReferenceFT,50), axis tight
pause

%do the same for the liver
figure(34),title('SEGMENT THE LIVER')
maskLiverFT=roipoly;
maskLiverFT=repmat(maskLiverFT,[1 1 dim3]);

% combine the liver and 19F signal masks
tempCombinedMask=maskLiverFT;
index=(imAlphaDataFT ==0);
tempCombinedMask(index)=imAlphaDataFT(index);
arrayLiverFT=image19FFT(tempCombinedMask==1);
% arrayLiverFT=arrayLiverFT(arrayLiverFT>7*NoiseStDevFT); %if needed remove extra ghosting
figure(343),hist(arrayLiverFT,50), axis tight
pause

%do the same for the tumor
figure(34)
imagesc(mip19FFTCor,[0 max(image19FFT(:))/8]), axis image off
title('SEGMENT THE TUMOR')
maskTumorFT=roipoly;
maskTumorFT=repmat(maskTumorFT,[1 1 dim3]);
% maskTumorFT(:,:,1:6)=0; maskTumorFT(:,:,end)=0; % if needed: null slices that are always far outside the brain

% combine the Tumor and 19F signal masks
tempCombinedMask=maskTumorFT;
index=(imAlphaDataFT ==0);
tempCombinedMask(index)=imAlphaDataFT(index);
arrayTumorFT=image19FFT(tempCombinedMask==1);
arrayTumorFT=arrayTumorFT(arrayTumorFT>0.15*max(arrayTumorFT));
figure(344),hist(arrayTumorFT/NoiseStDevFT,50), axis tight

%% check what one of the selected arrays looks like
tempimg=zeros(dim1,dim2,dim3);
tempimg(tempCombinedMask)=image19FFT(tempCombinedMask);
imagesc(squeeze(tempimg(:,:,round(end/2))))

%% calculate volumes, concentrations, integrals
referencePfcConc=190/10; % in mM; for example - 10 times diluted PFCE
SliceThickness=1;
voxelVolume=SliceThickness*(FoV19F/dimx19F/H1InterpFactor)^2; %in mm^3

volume1H=sum(maskTumor1H(:))*voxelVolume;
noiseFT=NoiseStDevFT;
volumeReferenceFT=length(arrayReferenceFT)*voxelVolume;
volumeTumorFT=length(arrayTumorFT)*voxelVolume;
volumeLiverFT=length(arrayLiverFT)*voxelVolume;

signalReferenceFT=mean(arrayReferenceFT); % average reference signal
signalLiverFT=mean(arrayLiverFT);
signalTumorFT=mean(arrayTumorFT);

if isnan(signalTumorFT)
    signalTumorFT=0;
end

snrTumorFT=signalTumorFT/noiseFT;
concTumorFT=signalTumorFT/signalReferenceFT*referencePfcConc; % mM
amountTumorFT=volumeTumorFT*concTumorFT;

mousename

%% print variable row for export and save workspace
clc
printst= [scandate ' ' mousename ' ' num2str(referencePfcConc) ' ' num2str(voxelVolume) ' ' num2str(volume1H) ' ' num2str(noiseCS) ' ' num2str(noiseFT) ' ' ...
    num2str(volumeReferenceFT) ' ' num2str(volumeTumorFT) ' ' num2str(volumeLiverFT) ' ' num2str(signalReferenceFT) ' ' num2str(signalTumorFT) ' ' num2str(signalLiverFT) ' ' num2str(concTumorFT) ' ' num2str(amountTumorFT) ' ' ]

% save workspace - WILL OVERWRITE!
save([savePath '\' scandate '_' mousename '_workspace'])

