% Demo of pair-wise non-rigid registration for myocardial T1 rho mapping
%
% Corresponding paper: "Endogenous assessment of myocardial injury with
% single-shot model-based non-rigid motion-corrected T1 rho mapping"
% To be published in Journal of Cardiovascular Magnetic Resonance, 2021
% 
% Author: Aurelien Bustin (aurelien.bustin@ihu-liryc.fr)
% IHU Liryc, 2021


%% Add codes to path

addpath(genpath('CODES/'));


%% Load ex vivo dataset

load('DATA/exvivo_images.mat');


%% Construct interpolator
% Construct a lookup table for faster interpolation

nbins = 2^16;
W = 2;
xList = -W/2 : W/(nbins-1) : W/2 ;
LookupTable              = 1 - abs(xList); % linear
Interpolator.LookupTable = LookupTable;
Interpolator.nbins       = nbins;
Interpolator.W           = W;


%% Run pair-wise non rigid registration for last 2 images

Images        = double(img(:,:,4:5));
[Ny, Nx, Nt]  = size(Images);

% Choose a reference (i.e. target) image for the registration
ref = 2;

%-- Registration parameters --

Param.display           = 1;                    % shows intermediate results
Param.ref               = ref;                  % index of the reference (i.e. target) image in the dynamic series
Param.lambda            = 1.e-3;                % smoothness weight of the displacement fields
Param.dx                = 1.6;                  % spatial resolution in mm/pixel
Param.dt                = 1.6;                  % temporal resolution in s/pixel
Param.V0                = Inf;                  % mm/s (weigth of the smoothness constraint in time; Inf=no constraint in time)
Param.ResolutionLevels	= [1/16 1/8 1/4 1/2];   % Possible to optimize displacements at full resolution or not
Param.NbLoops           = 4;                    % number of iterations at each resolution level
Param.Interpolator      = Interpolator;         % interpolator used to apply image transformations
Param.RegularizerOrder  = 1;                    % 1: minimize norm of the 1st order derivative of displacements
                                                % 2: minimize norm of the 2nd order derivative of displacements

%-- Run registration --
fprintf('Starting registration...\n')
tic;
[Ux, Uy, RegisteredImages] = pairwise_registration_nonrigid(Images, Param);

elapsed_time     = toc;
elapsed_time_min = floor(elapsed_time/60);
elapsed_time_sec = elapsed_time - elapsed_time_min*60;
fprintf('Total time: %d min %2.1f sec\n', elapsed_time_min, elapsed_time_sec );


%% Display motion-corrected images

figure,
subplot(231),
imshow(Images(:,:,1), []); title('Original 1')
subplot(232),
imshow(Images(:,:,2),[]); title('Original 2')
subplot(233),
imshowpair(Images(:,:,1), Images(:,:,2)); title('Overlay difference - Original')
subplot(234),
imshow(RegisteredImages(:,:,1), []); title('Registered 1')
subplot(235),
imshow(RegisteredImages(:,:,2),[]); title('Registered 2')
subplot(236),
imshowpair(RegisteredImages(:,:,1), RegisteredImages(:,:,2)); title('Overlay difference - Registered')


%% Interpolate motion fields (wraping operator) and display

U_t = zeros(Ny,Nx,1,2);
U_t(:,:,:,1) = Uy(:,:,1);
U_t(:,:,:,2) = Ux(:,:,1);
I_registered = interp_mex( Images(:,:,1), U_t, Interpolator.LookupTable, Interpolator.nbins, Interpolator.W, 0 );


figure,
subplot(231),
imshow(Images(:,:,1), []); title('Original 1')
subplot(232),
imshow(Images(:,:,2),[]); title('Original 2')
subplot(233),
imshowpair(Images(:,:,1), Images(:,:,2)); title('Overlay difference - Original')
subplot(234),
imshow(I_registered, []); title('Registered 1')
subplot(235),
imshow(Images(:,:,2),[]); title('Registered 2')
subplot(236),
imshowpair(I_registered, RegisteredImages(:,:,2)); title('Overlay difference - Registered')



