#Title: postSeg_function.R
#Type: Script
#Version: 1.0
#Date: 2014-03-07
#Author: Martial Sankar
#Maintainer: Martial Sankar <martial.sankar@unil.ch>
#Description: Suite of function for information extraction from segmented images
#License: GPL (>= 2)
#Depends: R (>= 2.14.0)
#OS_type: unix
#Depends on: EBImage, matlab, RColorBrewer
#URL: http://www3.unil.ch/wpmu/hardtkelab/


library("EBImage")
library("matlab", lib.loc = "~/local/libs/R_libs")
library("RColorBrewer")



#getFeaturesOverall, function that returns a dataFrame containing features for the cell objects
#            @imSeg, String or Image Object, Segmented images (ImageJ output)
#            @imRaw, String or Image Object, Raw image (used to generate overlayed segmentation images)
#            @rayThresh, Numeric, filters all objects that have higher radius
#            @methodExt, String, method for computing the cell extension either *geomean* (default) or *ratio* 
getFeaturesOverall <- function(imSeg, imRaw=NULL, rayThresh, methodExt)
{	
	# 1 Load cell Obj
	labs<-extractObjFromFiles(imSeg=imSeg,findObj=TRUE)	
	cat("## Number of Objects : ", max(labs), "\n")
	
	# 2 get cell features
	ftrsShape <- computeFeatures.shape(labs)
	ftrsMom <- computeFeatures.moment2(labs)
	
	# 3 calculate polar coord
	dfPolCoord <- getPolarCoord(im=labs, cartCoord=ftrsMom[,c(1,2)])	
	dfall <- as.data.frame(cbind(dfPolCoord, ftrsMom, ftrsShape))
	
	# 4 calculate ext parameters
	extV <- findExt(dfall,f=methodExt)
	cat(" Done !\n")
	
	# 5 calculate incl parameters
	center <- dim(labs)[1:2]/2
	inclV <- findInclScaler(dfall,center)
	cat(" Done !\n")	
	dfall <-as.data.frame( cbind(dfall, extV, inclV))
	
	# 6 radius threshold
	dfall<-as.data.frame(dfall)
	if(!is.null(rayThresh)){
		cat("# Filtering...\n")
		ix <- as.numeric(which(dfall$radiusV<rayThresh), arr.ind =T)
		cat("## Initial object count ::", nrow(dfall), "## Filtered object count ::",(nrow(dfall)-length(ix))," (", (nrow(dfall)-length(ix))/nrow(dfall)*100,"%)\n")
		ftrs <- as.data.frame(dfall[ix,])
		cat("# ...Done!\n")
	}else {
		ftrs <- dfall 
	}
	
	return(ftrs)
	
}



#ExtractObjFromFiles, load and extract objects from a segmented image file
#            @imSeg, String or Image Object, Segmented images (ImageJ output)
#            @findObj, Boolean, assign label to obect
extractObjFromFiles <-  function (imSeg, findObj = FALSE)
{	
	cat("# Load Images ... ")
	
	if(is.character(imSeg)){
		imCur <- readImage(imSeg)
		if(dim(imCur)[3]==1){
			imCur <- imCur
		}else {
			imCur <- imCur[,,1]
		}
	}else {imCur <- imSeg}
	cat("Done!\n")
	
	grayScale <- TRUE
	if(grayScale){
		colorMode(imCur) <- Grayscale
	}
	retObj <- imCur
	if(findObj){
		cat("# Extract objects...")
		labs<- bwlabel(imCur)
		cat(" Done !\n")
		if(grayScale){
			colorMode(labs) <- Grayscale
		}
		retObj <- labs
	}
	return(retObj)
}



#getPolarCoord,convert a table of the polar coordinates relative to the center of the section, takes as input
#            @imSeg, String or Image Object, Segmented images (ImageJ output)
#            @cartCoord, dataFrame, contains the cartesian coordinates (cell ids as rownames)
getPolarCoord <- function(im, cartCoord) 
{
	imCur <- extractObjFromFiles(im)	
	
	# 1 find center coord of the section
	xyhp <- dim(imCur)/2
	xy <- dim(imCur)
	xy0 <- c(1,1)
	
	xhalf <- xyhp[1]
	yhalf <- xyhp[2]
	xmax <- xy[1]
	ymax <- xy[2]
	
	
	# 2 Transform cartesian coordinate into polar
	xNew <- as.numeric(cartCoord[,1])-xhalf
	yNew <- as.numeric(cartCoord[,2])-yhalf
	radiusV <- sqrt(xNew^2+yNew^2)
	angleV <- atan(yNew/xNew)
	
	df <- as.matrix(cbind(xNew, yNew, radiusV, angleV))
	return(df)
	
}	


#findIncl, function that returns the inclination angle between the majors axis and the ray.
#           @tab, df, table containing the features for each objects
#           @centerCoord, Vector, contains x,y coordinates of the center of the cross-section
findInclScaler <- function(tab,centerCoord)
{	
	theta <- tab[,"m.theta"]
	psi <- matrix(0, ncol=1, nrow=nrow(tab))
	thetap <- theta
	b <- tab[,"m.cy"]-tab[,"m.cx"]*tan(thetap)
	xM <- (0-b)/tan(thetap)
	cmCoordX <- xM-tab[,"m.cx"]
	cmCoordY <- 0-tab[,"m.cy"]
	dcm <- sqrt(cmCoordX^2+cmCoordY^2)
	coCoordX <- centerCoord[1]-tab[,"m.cx"]
	coCoordY <- centerCoord[2]-tab[,"m.cy"]
	psi <- acos(((coCoordX*cmCoordX)+(coCoordY*cmCoordY))/(tab[,"radiusV"]*dcm))
	psi <- abs(psi-(pi/2))
	return(psi)
}


#findExt, function that return extensibility (ext) parameters
#           @tab, dataFrame, table containing the features for each objects
#           @f, String, method for computing the cell extension *geomean*, *ratio*, 
findExt <- function(tab, f)
{
	cat("# Compute cell extension ...")
	minorAxis <- tab[,7]*sqrt(1-tab[,8]^2)
	if(f=="geomean"){	
		ext <- sqrt(minorAxis*tab[,7])
	}else if (f == "ratio"){
		ext <- minorAxis/tab[,7]
	}
	return(ext)	
}

#dispColorMapSection, function that make a colorMap of cell and generate a mask of it (time consuming)
#                @ imSeg, String or Image Object, Segmented images (ImageJ output)
#                @tab, dataFrame, contains the cell features
#                @colIndexFeatures, Integer, the index of the cell features to map
#                @rayThresh, Numeric, filters all objects that have higher radius
#                @areaThresh, Numeric, filters all objects that have higher area
#                @suffix, String, suffix for filename
#                @paletteCol, String, brewer palette (ie, BuGn, Blues, RdYlGn ...)
dispColorMapSection <- function(imSeg, tab, colIndexFeatures, rayThresh=NULL, areaThresh=NULL ,suffix=NULL, paletteCol=NULL){
	# 1 load image and data features
	labs <- extractObjFromFiles(imSeg=imSeg,findObj=TRUE)
	if(is.character(tab)){
		tab <- read.delim(tab, h=T, sep = "\t", as.is=T)
	}else {
		tab<-as.data.frame(tab)
	}
	colnm <- names(tab)[colIndexFeatures]
	cat("#Generate ColorMap for *",colnm,"* feature\n")

	# 2 filter ray
	if(!is.null(rayThresh)){
		tab0<-tab
		ut <- which(as.numeric(tab$radiusV)<rayThresh)
		if(length(ut)>0){
			tab<-tab[ut,]
			ids <- row.names(tab)
			rm(tab0)#get cell ids
		}
	}else {
		ids <- row.names(tab)
	}
	
	# 3 filter area	
	if(!is.null(areaThresh)){
		ut <- which(as.numeric(tab$s.area)>areaThresh)
		if(length(ut)>0){
			tab0 <- tab
			tab<-tab[-ut,]
			ids <- row.names(tab0)[-ut]
			rm(tab0)#get cell ids
		}	
		
	}
	
	# 4 generate color Gradient
	if(is.null(paletteCol)){
		col <-brewer.pal(8,"RdYlGn")
	}else {
		col <-brewer.pal(8,paletteCol)
	}
	vfeat <- as.numeric(tab[,colIndexFeatures])
	ixFeat <- 1:length(vfeat)
	ixFeatSorted <- ixFeat[order(vfeat)]
	#vfeatSorted <- vfeat[ixFeatSorted]
	idsSorted <- ids[ixFeatSorted]
	colRampPalette <- colorRampPalette(col[1:length(col)],space ="Lab",interpolate="linear",bias=1)(length(idsSorted))
	cols <- c("black", rep("white",max(labs)))	
	cols <-matrix(cols[1+labs],nrow=dim(labs))
	colsSelect <- cols
	
	# 5 generate vizualisation
	ix4viz <- NULL
	for(i in 1:length(idsSorted)){		
		df <- which(labs==as.numeric(idsSorted[i]), arr.ind=TRUE)
		ix4viz <- rbind(ix4viz,df)
		colsSelect[ix4viz] <- colRampPalette[i]	
		ix4viz <- NULL
		
	}
	cat("... Done ! \n")
	
	# 6 save image
	iNew <- Image(colsSelect)	
	cat("# Creating Image... \n")
	tmpName <- paste("selected_",colnm,"_",format(Sys.time(), "%y%m%d%H%M%S"),"_",suffix,".tif",sep="")
	suffixN <- tmpName
	checkgrpnm <-  paste(gsub("segmented_0.tif","", imSeg) ,suffixN,sep="")
	writeImage(iNew, checkgrpnm, quality = 100)
	cat("## saved in : ", checkgrpnm, "\n")
	return(checkgrpnm)
	
}


#dispColorCircleMapSection, function that make a color circle map of the section with each circle weighted on their corresponding cell size
#                @tab, dataFrame, contains the cell features
#                @colIndexFeatures, Integer, the index of the cell features to map
#                @rayThresh, Numeric, filters all objects that have higher radius
#                @areaThresh, Numeric, filters all objects that have higher area
#                @suffix, String, suffix for filename
#                @paletteCol, String, brewer palette (ie, BuGn, Blues, RdYlGn ...)
dispColorCircleMapSection <- function(tab, colIndexFeatures, rayThresh=NULL, areaThresh=NULL ,suffix=NULL, paletteCol=NULL)
{
	# 1 data features
	if(is.character(tab)){
		tab <- read.delim(tab, h=T, sep = "\t", as.is=T)
	}else {
		tab<-as.data.frame(tab)
	}
	colnm <- names(tab)[colIndexFeatures]
	cat("#Generate ColorMap for *",colnm,"* feature\n")
	
	# 2 filter ray
	if(!is.null(rayThresh)){
		tab0<-tab
		ut <- which(as.numeric(tab$radiusV)<rayThresh)
		if(length(ut)>0){
			tab<-tab[ut,]
			ids <- row.names(tab)
			rm(tab0)#get cell ids
		}
	}else {
		ids <- row.names(tab)
	}
	
	# 3 filter area
	if(!is.null(areaThresh)){
		ut <- which(as.numeric(tab$s.area)>areaThresh)
		if(length(ut)>0){
			tab0 <- tab
			tab<-tab[-ut,]
			ids <- row.names(tab0)[-ut]
			rm(tab0)#get cell ids
		}	
		
	}
	
	# 4 generate color gradient
	if(is.null(paletteCol)){
		col <-brewer.pal(8,"RdYlGn")
		
	}else {
		col <-brewer.pal(8,paletteCol)
	}
	vfeat <- as.numeric(tab[,colIndexFeatures])
	ixFeat <- 1:length(vfeat)
	ixFeatSorted <- ixFeat[order(vfeat)]
	idsSorted <- ids[ixFeatSorted]
	colRampPalette <- colorRampPalette(col[1:length(col)],space ="Lab",interpolate="linear",bias=1)(length(idsSorted))
	wvector <-  (as.numeric(tab[,"s.area"])/mean(as.numeric(tab[,"s.area"])))[ixFeatSorted] # weight vectors 
	coords <- tab[ixFeatSorted,c("m.cx", "m.cy")]
	tmpName <- paste("selected_",colnm,"_",format(Sys.time(), "%y%m%d%H%M%S"),"_",suffix,".pdf",sep="")
	checkgrpnm <-  paste(gsub("segmented.tif","", imSeg) ,tmpName,sep="")
	
	# 5 Vizualize and save
	pdf(checkgrpnm, height = 16, width = 16)
	plot(coords[,1], coords[,2], pch = 16, cex = wvector, col = colRampPalette)
	dev.off()
	cat(checkgrpnm,"\n")

}

#mapCellID, function that localized the cells given their ids on the segmented image
#           @imSeg, Image Object
#           @id, numeric
#           @fnout, full path and name for the output image
mapCellID <- function(imSeg, ids, fnout)
{
	cols <- c("white", rep("black",max(imSeg)))
	cols <-matrix(cols[1+imSeg],nrow=dim(imSeg))
	colsSelect <- cols
	ix4viz <- NULL
	for(i in 1:length(ids)){		
		df <- which(imSeg==as.numeric(ids[i]), arr.ind=TRUE)
		ix4viz <- rbind(ix4viz,df)
		colsSelect[ix4viz] <-  rgb(0,204,0,55, maxColorValue=255)
		ix4viz <- NULL
	}
	iNew <- Image(colsSelect)
	writeImage(iNew, fnout)
	cat("# Image saved in :: ", fnout,"\n")
}



